/**
* \file: info_messages.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <string.h>

#include "ipc/info_messages.h"
#include "ipc/message_recvr.h"
#include "ipc/message_sendr.h"
#include "utils/automounter_types_internal.h"
#include "utils/logger.h"

//------------------------------------- private attributes ------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------
static error_code_t info_messages_serialize_part_info(message_buffer_t *msg_buffer,partition_info_t *part_info_dst,
		const partition_info_t *part_info_src, device_info_t *dev_info_dst,
		const device_info_t *dev_info_src);
static error_code_t info_messages_deserialize_part_info(message_buffer_t *msg_buffer,
		partition_info_message_t *msg_ptr);

static error_code_t info_messages_serialize_dev_info(message_buffer_t *msg_buffer,device_info_t *dev_info_dst,
		const device_info_t *dev_info_src);
static error_code_t info_messages_deserialize_dev_info(message_buffer_t *msg_buffer,device_info_message_t *msg_ptr);
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member definition ------------------------------------------------------
error_code_t info_messages_prepare_part_info_msg(message_buffer_t *msg_buffer,info_event_t event,
		const device_info_t *dev_info, const partition_info_t *part_info, int request_id)
{
	error_code_t result;
	partition_info_message_t *msg_data;
	result=message_sendr_prepare_message(msg_buffer, PARTITION_STATE_CHANGED_MSG,sizeof(partition_info_message_t),(void **)&msg_data);
	if (result==RESULT_OK)
	{
		msg_data->event=event;
		msg_data->request_id=request_id;
		result=info_messages_serialize_part_info(msg_buffer, &(msg_data->part_info), part_info,
				&(msg_data->device_info), dev_info);
	}
	return result;
}

error_code_t info_messages_extract_part_info_msg(message_buffer_t *msg_buffer, partition_info_message_t **msg_ptr)
{
	size_t data_size;

	if (message_recvr_get_msg_type(msg_buffer)!=PARTITION_STATE_CHANGED_MSG)
		return RESULT_INVALID;

	message_recvr_get_msg_data(msg_buffer, (void **)msg_ptr, &data_size);
	if (data_size!=sizeof(partition_info_message_t))
		return RESULT_INVALID;

	return info_messages_deserialize_part_info(msg_buffer, *msg_ptr);
}

error_code_t info_messages_prepare_dev_info_msg(message_buffer_t *msg_buffer,info_event_t event,
		const device_info_t *dev_info, int request_id)
{
	error_code_t result;
	device_info_message_t *msg_data;
	result=message_sendr_prepare_message(msg_buffer,DEVICE_STATE_CHANGED_MSG,sizeof(device_info_message_t),(void **)&msg_data);
	if (result==RESULT_OK)
	{
		msg_data->event=event;
		msg_data->request_id=request_id;
		result=info_messages_serialize_dev_info(msg_buffer, &(msg_data->device_info), dev_info);
	}
	return result;
}

error_code_t info_messages_extract_dev_info_msg(message_buffer_t *msg_buffer, device_info_message_t **msg_ptr)
{
	size_t data_size;

	if (message_recvr_get_msg_type(msg_buffer)!=DEVICE_STATE_CHANGED_MSG)
		return RESULT_INVALID;

	message_recvr_get_msg_data(msg_buffer,(void **)msg_ptr, &data_size);
	if (data_size!=sizeof(device_info_message_t))
		return RESULT_INVALID;

	return info_messages_deserialize_dev_info(msg_buffer, *msg_ptr);
}

error_code_t request_message_prepare_appinfo_msg(message_buffer_t *msg_buffer,const char *app_identifier,
		event_mask_t event_mask)
{
	error_code_t result;
	event_mask_t *event_mask_ptr;
	result=message_sendr_prepare_message(msg_buffer,SET_APPINFO_MSG,sizeof(event_mask_t),(void **)&event_mask_ptr);
	if (result==RESULT_OK)
	{
		*event_mask_ptr=event_mask;
		result=message_sendr_add_string_to_msg(msg_buffer, app_identifier);
	}

	return result;
}

error_code_t request_message_extract_appinfo_msg(message_buffer_t *msg_buffer, const char **app_identifier_ptr,
		event_mask_t **event_mask_ptr)
{
	size_t data_size;

	if (message_recvr_get_msg_type(msg_buffer)!=SET_APPINFO_MSG)
		return RESULT_INVALID;

	message_recvr_get_msg_data(msg_buffer, (void **)event_mask_ptr, &data_size);
	if (data_size!=sizeof(event_mask_t))
		return RESULT_INVALID;
	*app_identifier_ptr=message_recvr_get_first_msg_string(msg_buffer);

	return RESULT_OK;
}

//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------
static error_code_t info_messages_serialize_part_info(message_buffer_t *msg_buffer,partition_info_t *part_info_dst,
		const partition_info_t *part_info_src, device_info_t *dev_info_dst,
		const device_info_t *dev_info_src)
{
	error_code_t result;

	memcpy(dev_info_dst,dev_info_src,sizeof(device_info_t));
	memcpy(part_info_dst,part_info_src,sizeof(partition_info_t));

	//serialize the device strings
	result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->device_handler_id);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->device_type);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->identifier);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->interface_id);

	//serialize the partition strings
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, part_info_src->identifier);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, part_info_src->interface_id);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, part_info_src->mount_fs);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, part_info_src->mount_point);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, part_info_src->mount_src);

	return result;
}

static error_code_t info_messages_deserialize_part_info(message_buffer_t *msg_buffer,
		partition_info_message_t *msg_ptr)
{
	//deserialize the device strings
	msg_ptr->device_info.device_handler_id=message_recvr_get_first_msg_string(msg_buffer);
	msg_ptr->device_info.device_type=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->device_info.identifier=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->device_info.interface_id=message_recvr_get_next_msg_string(msg_buffer);

	//check them
	if (msg_ptr->device_info.device_handler_id==NULL 	||
		msg_ptr->device_info.device_type==NULL 			||
		msg_ptr->device_info.identifier==NULL 			||
		msg_ptr->device_info.interface_id==NULL)
		return RESULT_INVALID;

	//deserialize the partition strings
	msg_ptr->part_info.identifier=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->part_info.interface_id=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->part_info.mount_fs=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->part_info.mount_point=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->part_info.mount_src=message_recvr_get_next_msg_string(msg_buffer);

	//check them
	if (msg_ptr->part_info.identifier==NULL 		||
		msg_ptr->part_info.interface_id==NULL		||
		msg_ptr->part_info.mount_fs==NULL			||
		msg_ptr->part_info.mount_point==NULL		||
		msg_ptr->part_info.mount_src==NULL)
		return RESULT_INVALID;

	return RESULT_OK;
}

static error_code_t info_messages_serialize_dev_info(message_buffer_t *msg_buffer,device_info_t *dev_info_dst,
		const device_info_t *dev_info_src)
{
	error_code_t result;
	memcpy(dev_info_dst,dev_info_src,sizeof(device_info_t));

	//serialize the device strings
	result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->device_handler_id);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->device_type);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->identifier);
	if (result==RESULT_OK)
		result=message_sendr_add_string_to_msg(msg_buffer, dev_info_src->interface_id);

	return result;
}

static error_code_t info_messages_deserialize_dev_info(message_buffer_t *msg_buffer, device_info_message_t *msg_ptr)
{
	//deserialize the device strings
	msg_ptr->device_info.device_handler_id=message_recvr_get_first_msg_string(msg_buffer);
	msg_ptr->device_info.device_type=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->device_info.identifier=message_recvr_get_next_msg_string(msg_buffer);
	msg_ptr->device_info.interface_id=message_recvr_get_next_msg_string(msg_buffer);

	//check them
	if (msg_ptr->device_info.device_handler_id==NULL 	||
		msg_ptr->device_info.device_type==NULL 			||
		msg_ptr->device_info.identifier==NULL 			||
		msg_ptr->device_info.interface_id==NULL)
		return RESULT_INVALID;

	return RESULT_OK;
}
//---------------------------------------------------------------------------------------------------------------------
